<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Helpers\ReceiptHelper;
use App\Models\Donation;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Yajra\DataTables\Facades\DataTables;
use PDF;
use Rmunate\Utilities\SpellNumber;

class UserController extends Controller
{
    public function index()
    {
        $donation = Donation::all();
        return view('admin.donation.index', compact('donation'));
    }

    public function create()
    {
        return view('admin.donation.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'party_type' => 'required|string',
            'name' => 'required|string|max:255',
            'pan_no' => 'required|string',
            'address' => 'required|string|max:500',
            'payment_mode' => 'required|string|in:Cheque,DD,Cash,RTGS',
            'cheque_dd_no' => 'nullable|required_if:payment_mode,Cheque,DD|string',
            'reference_no' => 'nullable|required_if:payment_mode,RTGS|string',
            'drawn_on' => 'nullable|required_if:payment_mode,Cheque,DD,RTGS|date',
            'branch' => 'nullable|required_if:payment_mode,Cheque,DD,RTGS|string|max:255',
            'amount' => 'required|numeric|min:0.01'
        ],
        [
            'name.required' => 'Please enter your name.',
            'pan_no.required' => 'Please provide your PAN number.',
            'amount.required' => 'Please specify the amount.',
            "address.required" => 'Please enter your address.',
            "cheque_dd_no.required" => 'Please provide the cheque or DD number.',
            "reference_no.required" => 'Please provide the Reference number.',
            "drawn_on.required" => 'Please specify the bank name or date the :input was drawn.',
            "branch.required" => 'Please provide the branch name.'
            
        ]);
        
        $prefix = Config::get('constants.prefix.'.$request->party_type);
        $lastRcpt = Donation::select('receipt_no')->withTrashed()->where('party_type', $request->party_type)->orderBy('id', 'desc')->first();

        $latestReceiptNumber = '';
        if (isset($lastRcpt->receipt_no)) {
            $latestReceiptNumber = ReceiptHelper::increaseSeqNo($lastRcpt->receipt_no, $prefix);
        } else {
            $startFrom = Config::get("constants.prefix.{$prefix}_start_from");
            $latestReceiptNumber = $prefix . str_pad($startFrom, 6, '0', STR_PAD_LEFT);
        }

        $data = $request->all();
        $donation = new Donation();
        $donation->party_type = isset($data['party_type']) ? $data['party_type'] : NULL;
        $donation->name = isset($data['name']) ? $data['name'] : NULL;
        $donation->pan_no = isset($data['pan_no']) ? $data['pan_no'] : NULL;
        $donation->address = isset($data['address']) ? $data['address'] : NULL;
        $donation->payment_mode = isset($data['payment_mode']) ? $data['payment_mode'] : NULL;
        $donation->cheque_dd_no = isset($data['cheque_dd_no']) ? $data['cheque_dd_no'] : NULL;
        $donation->reference_no = isset($data['reference_no']) ? $data['reference_no'] : NULL;
        $donation->drawn_on = isset($data['drawn_on']) ? $data['drawn_on'] : NULL;
        $donation->branch = isset($data['branch']) ? $data['branch'] : NULL;
        $donation->amount = isset($data['amount']) ? $data['amount'] : NULL;
        $donation->receipt_no = $latestReceiptNumber;
        if ($donation->save()) {
            return redirect()->route('admin.donation.index') ->with('title', 'Success')->with('status', 'success')->with('message', ['Donation added successfully.']);
        }

        return redirect()->route('admin.donation.index', [
                'title' => 'Failed!',
                'status' => 'failed',
                'message' => 'failed.'
            ]);
    }

    public function edit(Donation $donation)
    {
        return view('admin.donation.edit', compact('donation'));
    }

    public function update(Request $request, Donation $donation)
    {
        $request->validate([
            'party_type' => 'required|string',
            'name' => 'required|string|max:255',
            'pan_no' => 'required|string',
            'address' => 'required|string|max:500',
            'payment_mode' => 'required|string|in:Cheque,DD,Cash,RTGS',
            'cheque_dd_no' => 'nullable|required_if:payment_mode,Cheque,DD|string',
            'reference_no' => 'nullable|required_if:payment_mode,RTGS|string',
            'drawn_on' => 'nullable|required_if:payment_mode,Cheque,DD,RTGS|date',
            'branch' => 'nullable|required_if:payment_mode,Cheque,DD,RTGS|string|max:255',
            'amount' => 'required|numeric|min:0.01'
        ],
        [
            'name.required' => 'Please enter your name.',
            'pan_no.required' => 'Please provide your PAN number.',
            'amount.required' => 'Please specify the amount.',
            "address.required" => 'Please enter your address.',
            "cheque_dd_no.required" => 'Please provide the cheque or DD or Reference number.',
            "reference_no.required" => 'Please provide the Reference number.',
            "drawn_on.required" => 'Please specify the bank name or date the :input was drawn.',
            "branch.required" => 'Please provide the branch name.'
            
        ]);

        $donation->update($request->all());
        return redirect()->route('admin.donation.index')->with('success', 'Donation updated successfully.');
    }

    public function destroy(Donation $donation)
    {
        $donation->delete();
        return redirect()->route('admin.donation.index')->with('success', 'Donation deleted successfully.');
    }

    public function delete($id)
    {
        $donation = Donation::find($id);

        return view('admin.donation.delete', compact('donation'));
    }

    public function getDonationsData(Request $request)
    {
        $donation = Donation::query();

        // If a party_type is selected, apply the filter
        if ($request->has('party_type') && ! empty($request->party_type)) {
            $donation->where('party_type', $request->party_type);
        }

        return DataTables::of($donation)
            ->addColumn('actions', function ($donation) {
                return '<button class="btn btn-sm btn-primary doner-edit">Edit</button>
                        <form action="'.route('admin.donation.destroy', $donation->id).'" method="POST" style="display:inline;">
                            '.csrf_field().method_field('DELETE').  '
                            <button type="submit" class="btn btn-sm btn-danger">Delete</button>
                        </form>';
            })
            ->rawColumns(['actions']) // Enable HTML rendering for the 'actions' column
            ->make(true);
    }

    public function downloadPDF($id)
    {
        $donation = Donation::findOrFail($id);
        
        if (empty($donation)) {
            return request()->route('admin.donation.index');
        }
        
        $data = [
            'receipt_no' => $donation->receipt_no,
            'party_type' => $donation->party_type,
            'name' => $donation->name,
            'pan_no' => $donation->pan_no,
            'address' => $donation->address,
            'payment_mode' => $donation->payment_mode,
            'cheque_dd_no' => $donation->cheque_dd_no,
            'drawn_on' => isset($donation->drawn_on) && !empty($donation->drawn_on) ? Carbon::parse($donation->drawn_on)->format('d/m/Y') : '',
            'branch' => $donation->branch,
            'amount' => round($donation->amount),
            'created_at' => $donation->created_at->format('d/m/Y'),
            'amount_in_words' => isset($donation->amount) && !empty($donation->amount) ? ucwords(SpellNumber::value(round($donation->amount))->toLetters()) : ''
        ];

        if ($donation->party_type == 'RVJP') {
            $view = 'rjvp-receipt';
        } else if ($donation->party_type == 'UECT'){
            $view = 'uect-receipt';
        } else {
            $view = 'receipt';
        }
        // Load view and generate PDF
        $pdf = PDF::loadView('admin.donation.'.$view, $data);

        // Download the generated PDF
        return $pdf->download('donation_receipt_' . $donation->receipt_no . '.pdf');
    }
}
